using System;
using System.Windows.Forms;
using System.Drawing;
using VNC.RFBDrawing;
using System.Threading;
using VNC.Config;

#region

// author: Dominic Ullmann, dominic_ullmann@swissonline.ch
// Version: 1.02
        
// VNC-Client for .NET
// Copyright (C) 2002 Dominic Ullmann

// author: Noh Seoung Hyun, gerranwizard@hotmail.com
// Version: 1.0

// VNC-Client for .NET Compact Framework
// Copyright (C) 2003 Noh Seoung Hyun

// This program is free software; 
// you can redistribute is and/or modify it under the terms of 
// the GNU General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.

#endregion

namespace VNC 
{
	/// <summary> Application main class </summary>
	public class VNCClient 
	{
		/// <summary> the port to connect to </summary>
		private int port = 5900;
		public int Port
		{
			get { return port; }
		}

		/// <summary> the server to connect to </summary>
		private string server = "";		
		public string Server
		{
			get { return server; }
		}
		/// <summary> the configuration information </summary>
		private VNCConfiguration config;
		/// <summary> the application entry point </summary>
		public static void Main(string[] args) 
		{
			// Starts message handling loop (events from main window) and shows main window
			VNCClient client = new VNCClient();
			
			try 
			{
				client.config = new VNCConfiguration();
			} 
			catch (Exception) 
			{
				// I will modify this region.
				Application.Exit();
			}

			client.getConnectionInformation();
			
			client.connect();
 		}

 		/// <summary> retrieve the connection Information </summary>
 		private void getConnectionInformation() 
		{
			// ask user for VNC-Server address and port
		    ConnectionForm conn = new ConnectionForm();  	
			
			if (conn.ShowDialog() == DialogResult.OK)
			{
				server = conn.getServer();
				port = conn.getPort();
				conn.Dispose();
			} 
			else 
			{
			    conn.Dispose();
				Console.WriteLine("Connect Aborted");
				// I will modify this region.
				Application.Exit();
		    }
 		}
 		
 		/// <summary> connect to the VNC-Server </summary>
 		private void connect() 
		{
			Console.WriteLine("Connect to: " + server + " on port: " + port);
			// create the surface
			RFBSurface surface = new RFBSurface(server, port, config);
				
			for (int i = 1; i <= config.NrOfViews; i++) 
			{
				connectView(surface);	
			}			
 		}
 		
 		/// <summary> create and connect a view to the surface </summary>
 		private void connectView(RFBSurface surface) 
		{
			ClientWin win = new ClientWin();
			RFBView view = new RFBView();
			// calling this to activate the view:
			view.connect(win, surface);
			win.registerView(view);
			try
			{
				win.Show();
				MessageBox.Show("initilizing screen");
			}
			catch (Exception e)
			{
				Console.WriteLine("--?" + e);
			}
			// Application.Run(win) is here for passing win parameter to this thread.
			Application.Run(win);
 		}		
	}

	/// <summary> one of the application windows </summary>
	/// <remarks> this Windows is a container for an RFB-View, an RFB-View
	/// is connected to this window using the registerView method. </remarks>
	public class ClientWin : Form 
	{
		private RFBView view;
		private String serverName;	
		private String serverIP;
		private String port;
		private String resolution;
		private String depth;
		private String protocolVerion;
		private Microsoft.WindowsCE.Forms.InputPanel inputPanel;
		private System.Windows.Forms.MainMenu mainMenu;
		private System.Windows.Forms.MenuItem connection;
		private System.Windows.Forms.MenuItem information;
		private System.Windows.Forms.MenuItem close;
		private System.Windows.Forms.MenuItem keys;
		private System.Windows.Forms.MenuItem Ctrl_Alt_Del;
		private System.Windows.Forms.MenuItem Ctrl_Down;
		private System.Windows.Forms.MenuItem Ctrl_Up;
		private System.Windows.Forms.MenuItem Alt_Down;
		private System.Windows.Forms.MenuItem Alt_Up;
		private System.Windows.Forms.MenuItem about;

		public ClientWin()
		{
			FormBorderStyle = FormBorderStyle.FixedDialog;

			this.Width = 246;
			this.Height = 295;

			this.connection = new MenuItem();
			this.information = new MenuItem();
			this.close = new MenuItem();
			this.keys = new MenuItem();
			this.Ctrl_Down = new MenuItem();
			this.Ctrl_Up = new MenuItem();
			this.Alt_Down = new MenuItem();
			this.Alt_Up = new MenuItem();
			this.about = new MenuItem();
			this.mainMenu = new System.Windows.Forms.MainMenu();
			this.inputPanel = new Microsoft.WindowsCE.Forms.InputPanel();

			// 
			// connection menuItem
			//
			this.connection.MenuItems.Add(this.information);
			this.connection.MenuItems.Add(this.close);
			this.connection.Text = "Connection";
			//
			// keys menuItem
			//
			this.keys.MenuItems.Add(this.Ctrl_Down);
			this.keys.MenuItems.Add(this.Ctrl_Up);
			this.keys.MenuItems.Add(this.Alt_Down);
			this.keys.MenuItems.Add(this.Alt_Up);
			this.keys.Text = "Keys";
			//
			// information menuItem
			//
			this.information.Text = "Information";
			this.information.Click +=new EventHandler(this.information_Click);
			// 
			// close menuItem
			//
			this.close.Text = "Close";
			this.close.Click += new System.EventHandler(this.close_Click);
			//
			// Ctrl-Down menuItem
			//
			this.Ctrl_Down.Text = "Ctrl-Down";
			this.Ctrl_Down.Click += new EventHandler(this.Ctrl_Down_Click);
			// 
			// Ctrl-Up menuItem
			//
			this.Ctrl_Up.Text = "Ctrl-Up";
			this.Ctrl_Up.Click += new EventHandler(this.Ctrl_Up_Click);
			//
			// Alt-Down menuItem
			//
			this.Alt_Down.Text = "Alt-Down";
			this.Alt_Down.Click += new EventHandler(this.Alt_Down_Click);
			//
			// Alt-Up menuItem
			//
			this.Alt_Up.Text = "Alt-Up";
			this.Alt_Up.Click += new EventHandler(this.Alt_Up_Click);
			//
			// about menutem
			//
			this.about.Text = "About";
			this.about.Click += new EventHandler(this.about_Click);
			//
			// mainMenu
			//
			this.mainMenu.MenuItems.Add(this.connection);
			this.mainMenu.MenuItems.Add(this.keys);
			this.mainMenu.MenuItems.Add(this.about);
			//
			// ClientWin Form
			//
			this.Menu = this.mainMenu;
		}

		private void information_Click(object sender, System.EventArgs e)
		{
			MessageBox.Show("Connected to: " + this.serverName
				+ "                       "
				+ "Host:" + this.serverIP
				+ "                       "
				+ "Port:" + this.port
				+ "                       "
				+ "Resolution:" + this.resolution
				+ "                       "
				+ "Depth:" + this.depth
				+ "                       "
				+ "Protocol Version:" + this.protocolVerion,
				"VNC connection info",
				System.Windows.Forms.MessageBoxButtons.OK,
				System.Windows.Forms.MessageBoxIcon.Asterisk,
				System.Windows.Forms.MessageBoxDefaultButton.Button1);
		}
				

		private void close_Click(object sender, System.EventArgs e)
		{
			this.view.ctrlDown = false;
			this.view.altDown = false;
			this.view.surface.closeConnection();
			Application.Exit();
		}

		private void Ctrl_Down_Click(object sender, System.EventArgs e)
		{
			this.view.ctrlDown = true;
		}

		private void Ctrl_Up_Click(object sender, System.EventArgs e)
		{
			this.view.ctrlDown = false;
		}

		private void Alt_Down_Click(object sender, System.EventArgs e)
		{
			this.view.altDown = true;
		}

		private void Alt_Up_Click(object sender, System.EventArgs e)
		{
			this.view.altDown = false;
		}

		private void about_Click(object sender, System.EventArgs e)
		{
			MessageBox.Show("VNC-Client for .NET Compact Framework"
				+ "                       "
				+ "Copyright (C) 2003"
				+ "                       "
				+ "Noh Seoung Hyun"
				+ "                       "
				+ "in Hanyang University"
				+ "                       "
				+ "gerranwizard@hotmail.com"
				+ "                       "
				+ "Thanks to Dominic Ullmann"
				+ "                       "
				+ "                       "
				+ "For more information see"
				+ "                       "
				+ "http://vnc.kldp.net"
				+ "                       "
				+ "Virtual Network Computing from ORL"
				+ "                       "
				+ "This program is under GPL",
				"About VNC-Client",
				System.Windows.Forms.MessageBoxButtons.OK,
				System.Windows.Forms.MessageBoxIcon.Asterisk,
				System.Windows.Forms.MessageBoxDefaultButton.Button1);

		}

		/// <summary> gets the servername of the vnc-server. </summary>
		/// <remarks> This name is sent during connection handshake. This window got this name from the RFBView it contains.
		/// Getting the value from this property makes only sense after a view is registered. </remarks>
		public String ServerName 
		{
			get { return serverName; }
		}

		/// <summary>
		/// register a view for beeing displayed in this window
		///	before calling this, view must be connected to a RFBSurface
		/// </summary>
		public void registerView(RFBView view)
		{
			if (!view.Connected) 
			{ 
				throw new Exception("only possible when view is connected"); 
			}
			this.view = view;
			this.view.Location = new System.Drawing.Point(0, 0);
		
			this.Controls.Add(this.view);
			this.KeyDown += new KeyEventHandler(this.view.viewKeyDown);
			this.KeyUp += new KeyEventHandler(this.view.viewKeyUp);
			// need notification on window closing
			Closing += new System.ComponentModel.CancelEventHandler(windowClosing);
			this.serverName = view.ServerName;
			Text = "Connected to: " + serverName;
			setServerInformation();
		}
		private void setServerInformation()
		{
			this.serverIP = this.view.surface.Server;
			this.port = "" + this.view.surface.Port;
			this.resolution = this.view.surface.BufferWidth + "x" + this.view.surface.BufferHeight;
			this.depth = "" + this.view.surface.Depth;
			this.protocolVerion = "3.3";
		}

		/// <summary> handles window-close events </summary>
		private void windowClosing(object sender, System.ComponentModel.CancelEventArgs e) 
		{
			// tell the view, that it's no longer drawable
			// we must call disconnectView, if view is no longer visible, otherwise connection to the remote server is not
			// correctly closed ...
			view.disconnectView();
		}
	}

	/// <summary> a from for asking for server-address and server-port </summary>
	public class ConnectionForm : Form {
		private Label info;
		private TextBox server;	
		private Button ok;
		private Button cancel;	
		private Microsoft.WindowsCE.Forms.InputPanel inputPanel;
	
		private System.Windows.Forms.MainMenu mainMenu;
	
		/// <summary> constructor for a connection form </summary>
		public ConnectionForm() {
			FormBorderStyle = FormBorderStyle.FixedDialog;
			Text = "Connect to Server";
			info = new Label();
			info.Text = "Please provide Server-Name";
			info.Location = new Point(10, 10);
			info.Size = new Size(200, 30);
			
			server = new TextBox();
			server.Text = "";
			server.ReadOnly = false;
			server.Location = new Point(10, 50);
			server.Size = new Size(200, 30);

			ok = new Button();
			ok.Location = new Point(10, 100);
			ok.DialogResult = DialogResult.OK;
			ok.Text = "Connect";
			
			cancel = new Button();
			cancel.Location = new Point(100, 100);
			cancel.DialogResult = DialogResult.Cancel;
			cancel.Text = "Abort";

			this.mainMenu = new System.Windows.Forms.MainMenu();
			this.inputPanel = new Microsoft.WindowsCE.Forms.InputPanel();
			this.Menu = this.mainMenu;
			
			Controls.Add(info);
			Controls.Add(server);
			Controls.Add(ok);
			Controls.Add(cancel);
		}
		
		/// <summary> gets the server-identifier part of the input </summary>
		public string getServer() 
		{
			if (server.Text.IndexOf(":") > 0) 
			{
				return server.Text.Substring(0,server.Text.IndexOf(":"));
			} 
			else 
			{	
				return server.Text;
			}
		}
		
		/// <summary> gets the server-port part of the input </summary>
		public int getPort() 
		{
			if (server.Text.IndexOf(":") > 0) 
			{
				return Int32.Parse(server.Text.Substring(server.Text.IndexOf(":") + 1));
			} 
			else 
			{	
				return 5900;
			}	
		}		
	}
}