#include "types.h"
#include "rovm.h"

#define OPCODE_FPUSH            0x0e
#define OPCODE_DPUSH            0x0f
#define OPCODE_CPUSH            0x10
#define OPCODE_HPUSH            0x11
#define OPCODE_IPUSH            0x12
#define OPCODE_SPUSH            0x13
#define OPCODE_DUP              0x59
#define OPCODE_IADD             0x60
#define OPCODE_CALL             0xb6
#define OPCODE_NEW              0xbb
#define OPCODE_NEWARRAY         0xbc

static void
_prealloc_ (rovm_t *r, size_t s)
{
  unsigned char *p;

  r->max_oplen = r->max_oplen + DEFAULT_OPCODE_LEN + s;
  p = (unsigned char *) apr_pcalloc (r->p, r->max_oplen);
  memcpy (p, r->op, r->oplen);

  r->op = p;
}

#define ROVM_OPFUNCS(NAME, TYPE)                \
  static inline void                            \
  NAME (rovm_t *r, TYPE val)                    \
  {                                             \
    TYPE *p = (TYPE *) (r->op + r->oplen);      \
                                                \
    r->oplen += sizeof (TYPE);                  \
    if (r->oplen >= r->max_oplen)               \
      _prealloc_ (r, 0);                        \
                                                \
    p[0] = val;                                 \
  }

ROVM_OPFUNCS (_opcode_, char);
ROVM_OPFUNCS (_uchar_, unsigned char);
ROVM_OPFUNCS (_ushort_, unsigned short);
ROVM_OPFUNCS (_uint_, unsigned int);
ROVM_OPFUNCS (_int_, int);

static void
_buf_ (rovm_t *r, const char *buf, size_t len)
{
  char *p = (char *) (r->op + r->oplen);

  r->oplen += len;
  if (r->oplen >= r->max_oplen)
    _prealloc_ (r, len);

  memcpy (p, buf, len);
}

static void 
opcode_ipush (rovm_t *r, int value)
{
  _opcode_ (r, OPCODE_IPUSH);
  _int_ (r, value);
}

static void 
opcode_spush (rovm_t *r, const char *buf, size_t len)
{
  _opcode_ (r, OPCODE_SPUSH);
  _uint_ (r, len);
  _buf_ (r, buf, len);
}

static void 
opcode_call (rovm_t *r, const char *mname, const char *mtype)
{
  _opcode_ (r, OPCODE_CALL);
  _uchar_ (r, strlen (mname));
  _buf_ (r, mname, strlen (mname));
  _ushort_ (r, strlen (mtype));
  _buf_ (r, mtype, strlen (mtype));
}

struct rovmops rovm_opcodes =
  {
    opcode_ipush,
    opcode_spush,
    opcode_call,
  };
