#include "GameCore.h"

MemPool::MemPool(LayerInfo* _pLayerInfo){
	pD3DDevice = _pLayerInfo->pD3DDevice;
	//setup material using Texture
	ZeroMemory( &lightTexturedMaterial, sizeof(D3DMATERIAL9) ); //general : 0
	lightTexturedMaterial.Diffuse.r = lightTexturedMaterial.Diffuse.g = lightTexturedMaterial.Diffuse.b = 1.0f;
	lightTexturedMaterial.Ambient.r = lightTexturedMaterial.Ambient.g = lightTexturedMaterial.Ambient.b = 1.0f;

	//setup material using Light source
	ZeroMemory( &lightEmissiveMaterial, sizeof(D3DMATERIAL9) );
	lightEmissiveMaterial.Diffuse.r = lightEmissiveMaterial.Diffuse.g = lightEmissiveMaterial.Diffuse.b = 1.0f;
	lightEmissiveMaterial.Ambient.r = lightEmissiveMaterial.Ambient.g = lightEmissiveMaterial.Ambient.b = 1.0f;
	lightEmissiveMaterial.Emissive.r = lightEmissiveMaterial.Emissive.g = lightEmissiveMaterial.Emissive.b = 1.0f;

	//create mesh especially sphere
	createSphere(20, 20);
	createCube(5, 5, 5);

	//create texture
	if(FAILED(D3DXCreateTextureFromFile(pD3DDevice, "Sun.bmp", &pSunTexture))){pSunTexture = NULL;}
	if(FAILED(D3DXCreateTextureFromFile(pD3DDevice, "Earth.bmp", &pEarthTexture))){pEarthTexture = NULL;}
	if(FAILED(D3DXCreateTextureFromFile(pD3DDevice, "Moon.bmp", &pMoonTexture))){pMoonTexture = NULL;}
	if(FAILED(D3DXCreateTextureFromFile(pD3DDevice, "Comet.bmp", &pCometTexture))){pCometTexture = NULL;}
	pCubeTexture = NULL;
}

MemPool::~MemPool(){
	if(pSphereVertexBuffer != NULL){pSphereVertexBuffer->Release(); pSphereVertexBuffer = NULL;}
	if(pSphereIndexBuffer != NULL){pSphereIndexBuffer->Release();  pSphereIndexBuffer = NULL;}
	if(pCubeVertexBuffer != NULL){pCubeVertexBuffer->Release(); pCubeVertexBuffer = NULL;}
	if(pCubeIndexBuffer != NULL){pCubeIndexBuffer->Release(); pCubeIndexBuffer = NULL;}

	if(pSunTexture != NULL){pSunTexture->Release(); pSunTexture=NULL;}
	if(pEarthTexture != NULL){pEarthTexture->Release(); pEarthTexture = NULL;}
	if(pMoonTexture != NULL){pMoonTexture->Release(); pMoonTexture = NULL;}
	if(pCometTexture != NULL){pCometTexture->Release(); pCometTexture = NULL;}
}

bool MemPool::createSphere(const int _rings, const int _segments){
	//Setup counts for this object
	numSphereVertex  = (_rings + 1) * (_segments + 1);	
	numSphereIndex = 2 * _rings * (_segments + 1);
	numShperePolygon = numSphereIndex - 2;

	if(FAILED(pD3DDevice->CreateVertexBuffer(numSphereVertex * sizeof(VertexCustom),
											0, D3DFVF_CUSTOM,D3DPOOL_DEFAULT,
											&pSphereVertexBuffer, NULL))){return false;}
	if(FAILED(pD3DDevice->CreateIndexBuffer(numSphereIndex * sizeof(WORD), 
											0, D3DFMT_INDEX16, D3DPOOL_MANAGED,
											&pSphereIndexBuffer, NULL))){return false;}
	WORD* pSphereIndex;
	VertexCustom* pSphereVertex; 
	WORD wVertexIndex = 0;
	int currentRing;
	int currentSegment;
	D3DXVECTOR3 vNormal;

	if(FAILED(pSphereVertexBuffer->Lock(0, 0, (void**)&pSphereVertex, 0))){return false;}
	if(FAILED(pSphereIndexBuffer->Lock(0, numSphereIndex, (void**)&pSphereIndex, 0))){return false;}

	//Establish constants used in sphere generation
	FLOAT fRingAngle = (D3DX_PI / _rings);
	FLOAT fSegAngle = (2.0f * D3DX_PI / _segments);

	//Generate the group of rings for the sphere
	for(currentRing = 0; currentRing < _rings + 1; currentRing++){
		FLOAT r0 = sinf(currentRing * fRingAngle);
		FLOAT y0 = cosf(currentRing * fRingAngle);

		//Generate the group of segments for the current ring
		for(currentSegment = 0; currentSegment < _segments + 1; currentSegment++){
			FLOAT x0 = r0 * sinf(currentSegment * fSegAngle);
			FLOAT z0 = r0 * cosf(currentSegment * fSegAngle);

			vNormal.x = x0;
			vNormal.y = y0;
			vNormal.z = z0;

			D3DXVec3Normalize(&vNormal, &vNormal);

			//Add one vertex to the strip which makes up the sphere
			pSphereVertex->x = x0;
			pSphereVertex->y = y0;
			pSphereVertex->z = z0;
			pSphereVertex->nx = vNormal.x;
			pSphereVertex->ny = vNormal.y;
			pSphereVertex->nz = vNormal.z;
			pSphereVertex->tu = 1.0f - ((FLOAT)currentSegment / (FLOAT)_segments);
			pSphereVertex->tv = (FLOAT)currentRing / (FLOAT)_rings;

			pSphereVertex++;
	
			//Add two indices except for the last ring 
			if(currentRing != _rings){
				*pSphereIndex = wVertexIndex; 
				pSphereIndex++;
		
				*pSphereIndex = wVertexIndex + (WORD)(_segments + 1); 
				pSphereIndex++;
		
				wVertexIndex++;
			}
		}
	}

	if(FAILED(pSphereIndexBuffer->Unlock())){return false;}
	if(FAILED(pSphereVertexBuffer->Unlock())){return false;}
	return true; 
}


bool MemPool::createCube(const float _width, const float _height, const float _depth){
	
	//Setup counts for this object
	numCubeVertex = 24;	
	numCubeIndex  = 36;
	numCubePolygon = 12;
	
	//create vertexbuffer
    if(FAILED(pD3DDevice->CreateVertexBuffer(numCubeVertex * sizeof(VertexCustom),0, D3DFVF_CUSTOM,
                                             D3DPOOL_DEFAULT, &pCubeVertexBuffer, NULL))){return false;}
	//create indexbuffer
	if(FAILED(pD3DDevice->CreateIndexBuffer(numCubeIndex * sizeof(WORD), 
											0, D3DFMT_INDEX16, D3DPOOL_MANAGED,
											&pCubeIndexBuffer, NULL))){return false;}
	
	//Set values for the index buffer
	WORD pCubeIndex[] = { 0, 1, 2, 3, 2, 1,	//Top
					      4, 5, 6, 7, 6, 5,	//Face 1
					      8, 9,10,11,10, 9,	//Face 2
					     12,13,14,15,14,13,	//Face 3
					     16,17,18,19,18,17,	//Face 4
					     20,21,22,23,22,21};	//Bottom
	   
	//Get a pointer to the index buffer indices and lock the index buffer
	WORD* pTempIndex;
	if(FAILED(pCubeIndexBuffer->Lock(0, sizeof(pCubeIndex), (void**)&pTempIndex, 0))){return false;}
	memcpy(pTempIndex, pCubeIndex, sizeof(pCubeIndex));
	if(FAILED(pCubeIndexBuffer->Unlock())){return false;}

	VertexCustom pCubeVertex[] = {	
		//Top Face
		{0.0f - (_width / 2), 0.0f + (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 1.0f,},		//Vertex 0
		{0.0f - (_width / 2), 0.0f + (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 0.0f,},		//Vertex 1
		{0.0f + (_width / 2), 0.0f + (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 1.0f,},		//Vertex 2
		{0.0f + (_width / 2), 0.0f + (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 0.0f,},		//Vertex 3
		
		//Face 1
		{0.0f - (_width / 2), 0.0f - (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 1.0f,},		//Vertex 4
		{0.0f - (_width / 2), 0.0f + (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 0.0f,},		//Vertex 5
		{0.0f + (_width / 2), 0.0f - (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 1.0f,},		//Vertex 6
		{0.0f + (_width / 2), 0.0f + (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 0.0f,},		//Vertex 7
		
		//Face 2
		{0.0f + (_width / 2), 0.0f - (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 1.0f,},		//Vertex 8
		{0.0f + (_width / 2), 0.0f + (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 0.0f,},		//Vertex 9
		{0.0f + (_width / 2), 0.0f - (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 1.0f,},		//Vertex 10
		{0.0f + (_width / 2), 0.0f + (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 0.0f,},		//Vertex 11
		
		//Face 3
		{0.0f + (_width / 2), 0.0f - (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 1.0f,},		//Vertex 12
		{0.0f + (_width / 2), 0.0f + (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 0.0f,},		//Vertex 13
		{0.0f - (_width / 2), 0.0f - (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 1.0f,},		//Vertex 14
		{0.0f - (_width / 2), 0.0f + (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 0.0f,},		//Vertex 15

		//Face 4
		{0.0f - (_width / 2), 0.0f - (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 1.0f,},		//Vertex 16
		{0.0f - (_width / 2), 0.0f + (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 0.0f,},		//Vertex 17
		{0.0f - (_width / 2), 0.0f - (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 1.0f,},		//Vertex 18
		{0.0f - (_width / 2), 0.0f + (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 0.0f,},		//Vertex 19

		//Bottom Face
		{0.0f + (_width / 2), 0.0f - (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 1.0f,},		//Vertex 20
		{0.0f + (_width / 2), 0.0f - (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 0.0f, 0.0f,},		//Vertex 21
		{0.0f - (_width / 2), 0.0f - (_height / 2), 0.0f - (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 1.0f,},		//Vertex 22
		{0.0f - (_width / 2), 0.0f - (_height / 2), 0.0f + (_depth / 2), 0.0f, 0.0f, 0.0f, 1.0f, 0.0f,},		//Vertex 23
	};

	WORD* pTempVertex;	
	if(FAILED(pCubeVertexBuffer->Lock(0, sizeof(pCubeVertex), (void**)&pTempVertex, 0))){return false;}
	memcpy(pTempVertex, pCubeVertex, sizeof(pCubeVertex));
	if(FAILED(pCubeVertexBuffer->Unlock())){return false;}
	return true;
}

void EventActor::setActor(MemPool* _pMemPool, const int _type){
	ActorType_ = _type;
	pD3DDevice_ = _pMemPool->pD3DDevice;
	pVertexBuffer_ = _pMemPool->pSphereVertexBuffer;
	pIndexBuffer_  = _pMemPool->pSphereIndexBuffer;
	numVertex_ = _pMemPool->numSphereVertex;
	numPolygon_ = _pMemPool->numShperePolygon;
	numIndex_ = _pMemPool->numSphereIndex;

	switch(ActorType_){
		case SUN:
            pTexture_ = _pMemPool->pSunTexture;
            pLightMaterial_ = &(_pMemPool->lightEmissiveMaterial);
			break;
		case EARTH:
            pTexture_ = _pMemPool->pEarthTexture;
			pLightMaterial_ = &(_pMemPool->lightTexturedMaterial);
			break;
		case MOON:
            pTexture_ = _pMemPool->pMoonTexture;
            pLightMaterial_ = &(_pMemPool->lightTexturedMaterial);
			break;

		case COMET:
            pTexture_ = _pMemPool->pCometTexture;
            pLightMaterial_ = &(_pMemPool->lightTexturedMaterial);
			break;

		case BOX:
			pVertexBuffer_ = _pMemPool->pCubeVertexBuffer;
			pLightMaterial_ = &(_pMemPool->lightTexturedMaterial);
			pIndexBuffer_  = _pMemPool->pCubeIndexBuffer;
			numVertex_ = _pMemPool->numCubeVertex;
			numPolygon_ = _pMemPool->numCubePolygon;
			numIndex_ = _pMemPool->numCubeIndex;

			pTexture_ = _pMemPool->pCubeTexture;
			break;
	}
}

void EventActor::action(){
	#define FROM_SUN  7.0f
	#define SEC_SUN  10.0f
	float nSec;
	float fAngle;
	D3DXMATRIXA16 matTrn, matScale, matAxis, matTrnFromEarth, matTrnFromSun,
	              matRotSun, matRotEarth, matRotMoon, matRot, matNonRot;

	switch(ActorType_){
		case SUN:
			nSec = 10.0f; //20ʿ 1 
			fAngle = ((timeGetTime())*(2.0f * D3DX_PI))/(1000.0f * nSec);
			D3DXMatrixTranslation(&matTrn, 0.0f, 0.0f, 0.0f);
			D3DXMatrixRotationY( &matRot, fAngle );
			matWorld_ = matTrn * matRot;
			break;
		case EARTH:
			D3DXMatrixScaling(&matScale,0.7f,0.7f,0.7f);
			nSec = 10.0f; // ӵ 10ʿ 1
			fAngle = ((timeGetTime())*(2.0f * D3DX_PI))/(1000.0f * nSec);
			D3DXMatrixRotationY( &matRotEarth, fAngle);
			D3DXMatrixRotationX( &matAxis, 23.5f * D3DX_PI/180);

			// ӵ, SEC_SUNʸ 1
			fAngle = ((timeGetTime())*(2.0f * D3DX_PI))/(1000.0f * SEC_SUN);
			D3DXMatrixRotationY( &matRotSun, fAngle );
			D3DXMatrixRotationY( &matNonRot, -fAngle ); // ϴ° 
			//¾κ Ÿ
			D3DXMatrixTranslation(&matTrnFromSun, FROM_SUN, 0.0f, 0.0f);
			matWorld_ = matScale * matRotEarth * matAxis * matNonRot * matTrnFromSun * matRotSun ;
			break;
		case MOON:
			D3DXMatrixScaling(&matScale,0.4f,0.4f,0.4f);
			nSec = 10.0f; //5ʿ 1  ӵ
			fAngle = ((timeGetTime())*(2.0f * D3DX_PI))/(1000.0f * nSec);
			D3DXMatrixRotationY( &matRotMoon, fAngle);
			//κ Ÿ
			D3DXMatrixTranslation(&matTrnFromEarth, 3.0f, 0.0f, 0.0f);
			nSec = 5; //5ʿ 1    ӵ
			fAngle = ((timeGetTime())*(2.0f * D3DX_PI))/(1000.0f * nSec);
			D3DXMatrixRotationY( &matRotEarth, fAngle);
			D3DXMatrixRotationY( &matNonRot, -fAngle); // ϴ° 
			fAngle = ((timeGetTime())*(2.0f * D3DX_PI))/(1000.0f * SEC_SUN);
			D3DXMatrixTranslation(&matTrnFromSun, FROM_SUN, 0.0f, 0.0f);
			D3DXMatrixRotationY( &matRotSun, fAngle );
			
			//matWorld = matRotMoon * matTrnFromEarth * matRotEarth * matTrnFromSun * matRotSun;
			matWorld_ = matScale * matRotMoon * matNonRot * matTrnFromEarth * matRotEarth * matTrnFromSun * matRotSun;
			break;
		case BOX:
			D3DXMatrixIdentity( &matWorld_ );
			break;
	}
}

void EventActor::show(){
	pD3DDevice_->SetStreamSource(0, pVertexBuffer_, 0, sizeof(VertexCustom));
	pD3DDevice_->SetFVF(D3DFVF_CUSTOM);
	pD3DDevice_->SetIndices(pIndexBuffer_);
	pD3DDevice_->SetMaterial(pLightMaterial_);
	pD3DDevice_->SetTransform( D3DTS_WORLD, &matWorld_ );

	if(ActorType_ == BOX){pD3DDevice_->DrawIndexedPrimitive(D3DPT_TRIANGLELIST, 0, 0, numVertex_, 0, numPolygon_);return;}

	if(pTexture_ != NULL){
		pD3DDevice_->SetTexture(0, pTexture_);
		pD3DDevice_->SetTextureStageState(0, D3DTSS_COLOROP, D3DTOP_MODULATE);
		pD3DDevice_->SetTextureStageState(0, D3DTSS_COLORARG1, D3DTA_TEXTURE);
		pD3DDevice_->SetTextureStageState(0, D3DTSS_COLORARG2, D3DTA_CURRENT);
	}else{ //texture is not set
		pD3DDevice_->SetTextureStageState(0, D3DTSS_COLOROP, D3DTOP_SELECTARG2);
		pD3DDevice_->SetTextureStageState(0, D3DTSS_COLORARG1, D3DTA_TEXTURE);
		pD3DDevice_->SetTextureStageState(0, D3DTSS_COLORARG2, D3DTA_CURRENT);
	}
	pD3DDevice_->DrawIndexedPrimitive(D3DPT_TRIANGLESTRIP, 0, 0, numVertex_, 0, numPolygon_);
}

GameCore::GameCore(LayerInfo* _pLayerInfo){
	mouseLx_ = 0.0f;
	mouseLy_ = 0.0f;
	mouseRx_ = 0.0f;
	mouseRy_ = 0.0f;
	mouseZ_  = 15.0f;
	pLayerInfo_ = _pLayerInfo;
	pD3DDevice_ = _pLayerInfo->pD3DDevice;
}

void GameCore::setLight1(){
	D3DLIGHT9 light_;
	ZeroMemory( &light_, sizeof(D3DLIGHT9) );
	light_.Type = D3DLIGHT_POINT;
	light_.Position = D3DXVECTOR3( 0.0f, 0.0f, 0.0f );
	light_.Range = 100.0f;
	light_.Diffuse.r = light_.Diffuse.g = light_.Diffuse.b = light_.Diffuse.a = 1.0f;
	light_.Attenuation0 = 0.2f;

	pD3DDevice_->SetLight( 0, &light_ );
	pD3DDevice_->LightEnable( 0, true );
	//pD3DDevice_->SetRenderState( D3DRS_FILLMODE, D3DFILL_WIREFRAME );//̾
	//pD3DDevice_->SetRenderState( D3DRS_FILLMODE, D3DFILL_SOLID );//ָ
	pD3DDevice_->SetRenderState( D3DRS_ZENABLE, true );
	pD3DDevice_->SetRenderState( D3DRS_LIGHTING, true ); //default : true
	pD3DDevice_->SetRenderState(D3DRS_AMBIENT,0x00404040);//ȯ
}


void GameCore::Camera1(){
	//view matrix
	D3DXVECTOR3 vEyePt( 0.0f, 20.0f, 0.0f );
	D3DXVECTOR3 vLookatPt( 0.0f, 0.0f, 0.0f );
	D3DXVECTOR3 vUpVec( 1.0f, 0.0f, 0.0f );
	D3DXMATRIXA16 matFirst, matView, matMoveX, matMoveY;

	D3DXMatrixRotationZ( &matMoveX, -mouseLx_/300.0f );
	D3DXMatrixRotationX( &matMoveY, -mouseLy_/300.0f );
	D3DXVec3TransformCoord( &vEyePt, &vEyePt, &(matMoveX * matMoveY));
	D3DXVec3TransformCoord( &vUpVec, &vUpVec, &(matMoveX * matMoveY));

	D3DXMatrixLookAtLH( &matView, &vEyePt, &vLookatPt, &vUpVec );
	pD3DDevice_->SetTransform( D3DTS_VIEW, &matView );

	//projection matrix
	D3DXMATRIXA16 matProj;

	//D3DXMatrixPerspectiveFovLH( &matProj, D3DX_PI/4, (float)4/3, 1.0f, 100.0f );
	D3DXMatrixPerspectiveFovLH( &matProj, D3DX_PI/4, 1, 1.0f, 100.0f );
	pD3DDevice_->SetTransform( D3DTS_PROJECTION, &matProj );
}

void GameCore::Camera2(){
	//view matrix
	D3DXVECTOR3 vEyePt( 0.0f, mouseZ_, 0.0f );
	D3DXVECTOR3 vLookatPt( 0.0f, 0.0f, 0.0f );
	D3DXVECTOR3 vUpVec( 1.0f, 0.0f, 0.0f );
	D3DXMATRIXA16 matFirst, matView, matMoveX, matMoveY;

	D3DXMatrixRotationX( &matMoveX, mouseRx_/300.0f );
	D3DXMatrixRotationZ( &matMoveY, -mouseRy_/300.0f );
	D3DXVec3TransformCoord( &vEyePt, &vEyePt, &(matMoveX * matMoveY));
	D3DXVec3TransformCoord( &vUpVec, &vUpVec, &(matMoveX * matMoveY));

	D3DXMatrixLookAtLH( &matView, &vEyePt, &vLookatPt, &vUpVec );
	pD3DDevice_->SetTransform( D3DTS_VIEW, &matView );

	//projection matrix
	D3DXMATRIXA16 matProj;

	D3DXMatrixPerspectiveFovLH( &matProj, D3DX_PI/4, (float)4/3, 1.0f, 100.0f );
	pD3DDevice_->SetTransform( D3DTS_PROJECTION, &matProj );
}

void GameCore::InputListener(){
	if(FAILED(pLayerInfo_->pDIDeviceKB->GetDeviceState(sizeof(keyBuffer_),(LPVOID)&keyBuffer_)))
		{pLayerInfo_->pDIDeviceKB->Acquire();}
	if(keyBuffer_[DIK_R] & 0x80){
		mouseLx_ = 0.0f;
		mouseLy_ = 0.0f;
		mouseRx_ = 0.0f;
		mouseRy_ = 0.0f;
		mouseZ_  = 15.0f;
	}

	if(pLayerInfo_->pDIDeviceMouse->GetDeviceState(sizeof(mouseState_),(LPVOID)&mouseState_) == DIERR_INPUTLOST)
		{pLayerInfo_->pDIDeviceMouse->Acquire();}
	if(mouseState_.rgbButtons[MOUSE_LBUTTON] & 0x80){
		mouseLx_ += mouseState_.lX;
		mouseLy_ += mouseState_.lY;
	}
	if(mouseState_.rgbButtons[MOUSE_RBUTTON] & 0x80){
		mouseRx_ += mouseState_.lX;
		mouseRy_ += mouseState_.lY;
	}

	mouseZ_ += mouseState_.lZ/100;
	/*if(mouseState_.rgbButtons[MOUSE_MBUTTON] & 0x80){
		mouseZ_ += mouseState_.lZ;
	}*/
	//if(mouseState_.rgbButtons[MOUSE_LBUTTON] & 0x80){mouseLy_ += mouseState_.ly;}
}
	

void GameCore::run(){
	LPDIRECT3DSURFACE9 pRenderSurface = NULL;
	LPDIRECT3DTEXTURE9 pRenderTexture = NULL;
	pD3DDevice_->CreateTexture(256,256,1,D3DUSAGE_RENDERTARGET,D3DFMT_R5G6B5,D3DPOOL_DEFAULT,&pRenderTexture,NULL);
	pRenderTexture->GetSurfaceLevel(0,&pRenderSurface);

	LPDIRECT3DSURFACE9 pBackBuffer = NULL;
	pD3DDevice_->GetRenderTarget(0,&pBackBuffer);


	MemPool aMemPool(pLayerInfo_);
	someActors[SUN].setActor(&aMemPool, SUN);
	someActors[EARTH].setActor(&aMemPool, EARTH);
	someActors[MOON].setActor(&aMemPool, MOON);
	someActors[BOX].setActor(&aMemPool, BOX);

	MSG msg;
	ZeroMemory( &msg, sizeof(msg) );
	while( msg.message != WM_QUIT ){
		if( PeekMessage( &msg, NULL, 0U, 0U, PM_REMOVE ) ){
			TranslateMessage( &msg );
			DispatchMessage( &msg );
		}else{

			InputListener();

			pD3DDevice_->SetRenderTarget(0,pRenderSurface);
			pD3DDevice_->Clear( 0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER, D3DCOLOR_XRGB(0,0,0), 1.0f, 0 );
			//start
			pD3DDevice_->BeginScene();
			setLight1();
			Camera1();
			someActors[SUN].action();
			someActors[SUN].show();
			someActors[EARTH].action();
			someActors[EARTH].show();
			someActors[MOON].action();
			someActors[MOON].show();
			//end
			pD3DDevice_->EndScene();

			//other rendering
			pD3DDevice_->SetRenderTarget(0,pBackBuffer);
			pD3DDevice_->Clear(0,NULL,D3DCLEAR_TARGET | D3DCLEAR_ZBUFFER,D3DCOLOR_XRGB(100,100,100),1.0f,0);
			pD3DDevice_->SetRenderState( D3DRS_LIGHTING, false); //default : true

			pD3DDevice_->BeginScene();
			Camera2();
			someActors[BOX].action();
			pD3DDevice_->SetTexture(0,pRenderTexture);
			someActors[BOX].show();
			pD3DDevice_->EndScene();

			pD3DDevice_->Present( NULL, NULL, NULL, NULL );
		}
	}

	if(pRenderSurface != NULL){pRenderSurface->Release(); pRenderSurface = NULL;}
	if(pRenderTexture != NULL){pRenderTexture->Release(); pRenderTexture = NULL;}
	if(pBackBuffer != NULL){pBackBuffer->Release(); pBackBuffer = NULL;}
}
