package net.kldp.beat.action;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletException;

import net.kldp.beat.annotation.After;
import net.kldp.beat.annotation.Beat;
import net.kldp.beat.annotation.Before;
import net.kldp.beat.annotation.BeforeResult;
import net.kldp.beat.annotation.Result;
import net.kldp.beat.annotation.Results;
import net.kldp.beat.exception.InterceptorException;
import net.kldp.beat.interceptor.AfterInterceptor;
import net.kldp.beat.interceptor.BeforeInterceptor;
import net.kldp.beat.interceptor.Interceptor;
import net.kldp.beat.interceptor.SystemInterceptor;
import net.kldp.beat.interceptor.UserInterceptor;
import net.kldp.beat.web.interceptor.InterceptorFactory;

public class InterceptorStack {

	private List<Result> results = new ArrayList<Result>();
	private Map<UserInterceptor, Annotation> before = new HashMap<UserInterceptor, Annotation>();
	private Map<UserInterceptor, Annotation> after = new HashMap<UserInterceptor, Annotation>();
	private Map<SystemInterceptor, Annotation> system = new HashMap<SystemInterceptor, Annotation>();;
	private Method beforeMethod;
	private Method afterMethod;
	private Method resultBeforeMethod;

	public InterceptorStack(Object action) throws ServletException {
		for (Annotation annotation : getAnnotations(action)) {
			classifyAnnotations(annotation);
		}
		classifyMethods(action.getClass().getMethods());
	}

	private void classifyMethods(Method[] methods) {
		for (Method method : methods) {
			Before before = method.getAnnotation(Before.class);
			After after = method.getAnnotation(After.class);
			BeforeResult resultBefore = method.getAnnotation(BeforeResult.class);
			if (before != null && beforeMethod == null) {
				this.beforeMethod = method;
			} else if (resultBefore != null && resultBeforeMethod == null) {
				this.resultBeforeMethod = method;
			} else if (after != null && afterMethod == null) {
				this.afterMethod = method;
			}
		}
	}

	private void classifyAnnotations(Annotation annotation) {
		if (annotation instanceof Result) {
			results.add((Result) annotation);
		} else if (annotation instanceof Results) {
			Results _results = (Results) annotation;
			for (Result result : _results.value()) {
				results.add(result);
			}
		} else {
			classifyInterceptors(annotation);
		}
	}

	private void classifyInterceptors(Annotation annotation) {
		try {
			Interceptor interceptor = InterceptorFactory.getInterceptor(annotation);
			if (interceptor instanceof SystemInterceptor) {
				system.put((SystemInterceptor) interceptor, annotation);
			} else {
				if (interceptor instanceof BeforeInterceptor) {
					before.put((UserInterceptor) interceptor, annotation);
				}
				if (interceptor instanceof AfterInterceptor) {
					after.put((UserInterceptor) interceptor, annotation);
				}
			}
		} catch (InterceptorException e1) {
			e1.printStackTrace();
		}
	}

	public static ArrayList<Annotation> getAnnotations(Object action) {
		ArrayList<Annotation> annotations = new ArrayList<Annotation>();
		for (Annotation annotation : action.getClass().getAnnotations()) {
			if (isBeat(annotation)) {
				annotations.add(annotation);
			}
		}
		for (Class<?> clazz : action.getClass().getInterfaces()) {
			for (Annotation annotation : clazz.getAnnotations()) {
				annotations.add(annotation);
			}
		}
		return annotations;
	}

	public static boolean isBeat(Annotation annotation) {
		for (Annotation anno : annotation.annotationType().getAnnotations()) {
			if (anno instanceof Beat)
				return true;
		}
		return false;
	}

	public List<Result> getResults() {
		return results;
	}

	public boolean hasResults() {
		return results.size() > 0 ? true : false;
	}

	public Result getResult(String resultString) {
		for (Result result : results) {
			if (result.name().equals(resultString))
				return result;
		}
		return null;
	}

	public Map<SystemInterceptor, Annotation> getSystemInterceptors() {
		return system;
	}

	public Map<UserInterceptor, Annotation> getBeforeInterceptors() {
		return before;
	}

	public Map<UserInterceptor, Annotation> getAfterInterceptors() {
		return after;
	}

	public Method getBeforeMethod() {
		return beforeMethod;
	}

	public Method getAfterMethod() {
		return afterMethod;
	}

	public Method getBeforeResultMethod() {
		return resultBeforeMethod;
	}
}